/*! \file tlk_cert_errors.h 
**********************************************************************************
*Title:                         Discretix Certificate Handling Toolkit source file
*
* Filename:                     tlk_cert_errors.h
*
* Project, Target, subsystem:   Toolkit, Certificate Handling
* 
* Created:                      29.01.2008
*
* Modified:                     29.01.2008
*
* \Author                       Ira Boguslavsky
*
* \Remarks
*           Copyright (C) 2008 by Discretix Technologies Ltd. All Rights reserved.
**********************************************************************************/
#ifndef _TLK_CERT_ERRORS_H_
#define _TLK_CERT_ERRORS_H_
#ifdef __cplusplus
extern "C" {
#endif

    /*! \brief Certificate Handling Toolkit API Return Codes base                   **/
#define TLK_CERT_RC_ERROR_BASE                           (0xBD000000UL)

    /*! \brief Operation completed successfully                                     **/
#define TLK_CERT_RC_OK                                   (0x00000000UL)


    /*! \brief The error range number assigned to each module on its specified layer **/
#define TLK_CERT_ERROR_MODULE_RANGE                      (0x00000100UL)

    /*! \brief Defines the module index for error mapping **/
#define TLK_CERT_API_ERROR_IDX                           (0x00UL)
#define TLK_CERT_ASN1_ERROR_IDX                          (0x001L)
#define TLK_CERT_X509_ERROR_IDX                          (0x002L)
#define TLK_CERT_OCSP_ERROR_IDX                          (0x003L)
#define TLK_CERT_X509_UTIL_ERROR_IDX                     (0x004L)
#define TLK_CERT_OCSP_UTIL_ERROR_IDX                     (0x005L)
#define TLK_CERT_BUFF_UTIL_ERROR_IDX                     (0x006L)
#define TLK_CERT_1667_ERROR_IDX                          (0x007L)
#define TLK_CERT_DER_UTIL_ERROR_IDX                      (0x008L)
    /*           Defining the error spaces for each module on each layer              */
    /* ------------------------------------------------------------------------------ */

    /* API module on the Toolkit Certificate Handling layer base address - 0xBD000000 */
#define TLK_CERT_API_MODULE_ERROR_BASE              (TLK_CERT_RC_ERROR_BASE + \
    TLK_CERT_ERROR_MODULE_RANGE * TLK_CERT_API_ERROR_IDX)

    /* ASN1 module on the Toolkit Certificate Handling layer base address - 0xBD000100 */
#define TLK_CERT_ASN1_MODULE_ERROR_BASE             (TLK_CERT_RC_ERROR_BASE + \
    TLK_CERT_ERROR_MODULE_RANGE * TLK_CERT_ASN1_ERROR_IDX)
    /* X.509 module on the Toolkit Certificate Handling layer base address - 0xBD000200 */
#define TLK_CERT_X509_MODULE_ERROR_BASE             (TLK_CERT_RC_ERROR_BASE + \
    TLK_CERT_ERROR_MODULE_RANGE * TLK_CERT_X509_ERROR_IDX)

    /* OCSP module on the Toolkit Certificate Handling layer base address - 0xBD000300 */
#define TLK_CERT_OCSP_MODULE_ERROR_BASE             (TLK_CERT_RC_ERROR_BASE + \
    TLK_CERT_ERROR_MODULE_RANGE * TLK_CERT_OCSP_ERROR_IDX)

    /* X.509 Utilities module on the Toolkit Certificate Handling layer base address - 0xBD000400 */
#define TLK_CERT_X509_UTIL_MODULE_ERROR_BASE        (TLK_CERT_RC_ERROR_BASE + \
    TLK_CERT_ERROR_MODULE_RANGE * TLK_CERT_X509_UTIL_ERROR_IDX)

    /* OCSP Utilities module on the Toolkit Certificate Handling layer base address - 0xBD000500 */
#define TLK_CERT_OCSP_UTIL_MODULE_ERROR_BASE        (TLK_CERT_RC_ERROR_BASE + \
    TLK_CERT_ERROR_MODULE_RANGE * TLK_CERT_OCSP_UTIL_ERROR_IDX)

    /* Buffer Reader Utilities module on the Toolkit Certificate Handling layer base address - 0xBD000600 */
#define TLK_CERT_BUFF_UTIL_MODULE_ERROR_BASE        (TLK_CERT_RC_ERROR_BASE + \
    TLK_CERT_ERROR_MODULE_RANGE * TLK_CERT_BUFF_UTIL_ERROR_IDX)

    /* IEEE 1667 module on the Toolkit Certificate Handling layer base address - 0xBD000700 */
#define TLK_CERT_BUFF_1667_MODULE_ERROR_BASE        (TLK_CERT_RC_ERROR_BASE + \
    TLK_CERT_ERROR_MODULE_RANGE * TLK_CERT_1667_ERROR_IDX)

    /* DER module on the Toolkit Certificate Handling layer base address - 0xBD000800 */
#define TLK_CERT_BUFF_DER_MODULE_ERROR_BASE        (TLK_CERT_RC_ERROR_BASE + \
    TLK_CERT_ERROR_MODULE_RANGE * TLK_CERT_DER_UTIL_ERROR_IDX)

    /************************************************************************/
    /* Error codes for different modules                                    */
    /************************************************************************/
    /************************************************************************/
    /* API Module Errors  - 0xBD000000                                      */
    /************************************************************************/
    /*! \brief Operation failed                                         **/
#define TLK_CERT_RC_ERROR_API_FAIL                                        (TLK_CERT_API_MODULE_ERROR_BASE+0x01)

    /*! \brief Operation failed -   Null pointer received as input      **/
#define TLK_CERT_RC_ERROR_API_NULL_POINTER                                (TLK_CERT_API_MODULE_ERROR_BASE+0x02)

    /*! \brief Operation failed -   Insufficient workspace given by user **/
#define TLK_CERT_RC_ERROR_WORKSPACE                                       (TLK_CERT_API_MODULE_ERROR_BASE+0x03)

    /*! \brief Operation failed -   Unaligned workspace given by user   **/
#define TLK_CERT_RC_ERROR_WORKSPACE_NOT_ALIGN                             (TLK_CERT_API_MODULE_ERROR_BASE+0x04)

    /*! \brief Operation failed -   No certificate supplied by user    **/
#define TLK_CERT_RC_ERROR_NO_SUPPLIED_CERTIFICATES                        (TLK_CERT_API_MODULE_ERROR_BASE+0x05)

    /*! \brief Operation failed -   The length of given certificate is incorrect **/
#define TLK_CERT_RC_ERROR_CERTIFICATE_LENGTH_INVALID                      (TLK_CERT_API_MODULE_ERROR_BASE+0x06)

    /*! \brief Operation failed -   Context object integrity compromised **/
#define TLK_CERT_RC_ERROR_INTEGRITY                                       (TLK_CERT_API_MODULE_ERROR_BASE+0x07)

    /*! \brief Operation failed -   Certificate ID compute failed because of CRYS_HASH problems **/
#define TLK_CERT_RC_ERROR_API_CERT_ID_COMPUTE_FAILED                      (TLK_CERT_API_MODULE_ERROR_BASE+0x08)

    /*! \brief Operation failed -    Context Object MAC calculation failed because of CRYS_HASH problems **/
#define TLK_CERT_RC_ERROR_API_CONTEXT_OBJ_MAC_CALCULATE_FAILED            (TLK_CERT_API_MODULE_ERROR_BASE+0x09)

    /*! \brief Operation failed -    Random nonce part generation of nonce failed **/
#define TLK_CERT_RC_ERROR_API_NONCE_PART_GENERATION_FAILED                (TLK_CERT_API_MODULE_ERROR_BASE+0x0A)

    /*! \brief Operation failed -    Nonce structure MAC calculation failed because of CRYS_HASH problems **/
#define TLK_CERT_RC_ERROR_API_NONCE_MAC_CALCULATION_FAILED                (TLK_CERT_API_MODULE_ERROR_BASE+0x0B)

    /*! \brief Operation failed -    Failed on calculation of certificate identity because of CRYS_HASH problems**/
#define TLK_CERT_RC_ERROR_API_CERT_IDENTITY_CALC_FAILED                   (TLK_CERT_API_MODULE_ERROR_BASE+0x0C)

    /*! \brief Operation failed -    RVO MAC calculation failed because of CRYS_HASH problems **/
#define TLK_CERT_RC_ERROR_API_RVO_MAC_CALCULATE_FAILED                    (TLK_CERT_API_MODULE_ERROR_BASE+0x0D)


    /************************************************************************/
    /* ASN1 Module Errors - 0xBD000100                                      */
    /************************************************************************/
    /*! \brief Operation failed -   ASN1 Parser got invalid arguments **/
#define TLK_CERT_RC_ERROR_ASN1_BAD_ARGUMENTS                              (TLK_CERT_ASN1_MODULE_ERROR_BASE+0x01)

    /*! \brief Operation failed -   ASN1 Parser supposed to get specific structure of the item **/
#define TLK_CERT_RC_ERROR_ASN1_ITEM_NOT_FOUND                             (TLK_CERT_ASN1_MODULE_ERROR_BASE+0x02)

    /*! \brief Operation failed -   ASN1 Parser got invalid tag of one of the fields in the structure **/
#define TLK_CERT_RC_ERROR_ASN1_OVERFLOW                                   (TLK_CERT_ASN1_MODULE_ERROR_BASE+0x03)

    /*! \brief Operation failed -   ASN1 Parser got unexpected item class **/
#define TLK_CERT_RC_ERROR_ASN1_UNEXPECTED_ITEM_CLASS                      (TLK_CERT_ASN1_MODULE_ERROR_BASE+0x04)

    /*! \brief Operation failed -   ASN1 Parser got unexpected item encoding **/
#define TLK_CERT_RC_ERROR_ASN1_UNEXPECTED_ITEM_ENCODING                   (TLK_CERT_ASN1_MODULE_ERROR_BASE+0x05)

    /*! \brief Operation failed -   ASN1 Parser got unexpected item tag **/
#define TLK_CERT_RC_ERROR_ASN1_UNEXPECTED_ITEM_TAG                        (TLK_CERT_ASN1_MODULE_ERROR_BASE+0x06)

    /*! \brief Operation failed -       ASN1 Parser got invalid sequence structure **/
#define TLK_CERT_RC_ERROR_ASN1_INVALID_SEQ_STRUCTURE                      (TLK_CERT_ASN1_MODULE_ERROR_BASE+0x07)

    /*! \brief Operation failed -       ASN1 Parser got unexpected object identifier **/
#define TLK_CERT_RC_ERROR_ASN1_UNEXPECTED_OID                             (TLK_CERT_ASN1_MODULE_ERROR_BASE+0x08)

    /************************************************************************/
    /* X509 Module Errors - 0xBD000200                                      */
    /************************************************************************/

    /*! \brief Operation failed -    X.509 Parser got unsupported certificate version **/
#define TLK_CERT_RC_ERROR_X509_UNSUPPORTED_VERSION                         (TLK_CERT_X509_MODULE_ERROR_BASE+0x01)

    /*! \brief Operation failed -    X.509 Parser got too many extended key usages **/
#define TLK_CERT_RC_ERROR_X509_TOO_MANY_EXTENDED_KEY_USAGES                (TLK_CERT_X509_MODULE_ERROR_BASE+0x02)

    /*! \brief Operation failed -    X.509 Parser got unrecognized critical extension **/
#define TLK_CERT_RC_ERROR_X509_UNRECOGNIZED_CRITICAL_EXTENSION             (TLK_CERT_X509_MODULE_ERROR_BASE+0x03)

    /*! \brief Operation failed -    X.509 Parser got invalid time structure (buffer, the buffer length incorrect, etc)**/
#define TLK_CERT_RC_ERROR_X509_INVALID_TIME                                (TLK_CERT_X509_MODULE_ERROR_BASE+0x04)

    /*! \brief Operation failed -    X.509 Parser successfully recognized the signature field, but its size was incorrect **/
#define TLK_CERT_RC_ERROR_OCSP_RESPONSE_UNSUPPORTED_SIGNATURE_SIZE         (TLK_CERT_X509_MODULE_ERROR_BASE+0x05)

    /************************************************************************/
    /* OCSP UTILS Module Errors  - 0xBD000300                               */
    /************************************************************************/

    /*! \brief Operation failed -    OCSP Parser successfully recognized the that extension is critical, but not kind of the extension **/
#define TLK_CERT_RC_ERROR_OCSP_UNRECOGNIZED_CRITICAL_EXTENSION             (TLK_CERT_OCSP_MODULE_ERROR_BASE+0x01)

    /*! \brief Operation failed -        The structure of sequence is invalid **/
#define TLK_CERT_RC_ERROR_OCSP_INVALID_SEQ_STRUCTURE                       (TLK_CERT_OCSP_MODULE_ERROR_BASE+0x02)

    /*! \brief Operation failed -        The supplied response status is not successful **/
#define TLK_CERT_RC_ERROR_OCSP_RESPONSE_IS_NOT_SUCCESSFULL                 (TLK_CERT_OCSP_MODULE_ERROR_BASE+0x03)

    /*! \brief Operation failed -        The type of response is differ from Basic. We support only Basic OCSP Response types.**/
#define TLK_CERT_RC_ERROR_OCSP_RESPONSE_UNSUPPORTED_TYPE                   (TLK_CERT_OCSP_MODULE_ERROR_BASE+0x04)

    /*! \brief Operation failed -        The type of response id differ from V1.We support only v1 responses**/
#define TLK_CERT_RC_ERROR_OCSP_RESPONSE_UNSUPPORTED_VERSION                (TLK_CERT_OCSP_MODULE_ERROR_BASE+0x05)

    /************************************************************************/
    /* X.509 Utils Module Errors - 0xBD000400                               */
    /************************************************************************/

    /*! \brief Operation failed -    Subject name of the current ("parent") certificate is differ from the issuer name of the "child" certificate**/
#define TLK_CERT_RC_ERROR_X509_UTIL_SUBJECT_NAME_INVALID                   (TLK_CERT_X509_UTIL_MODULE_ERROR_BASE+0x01)

    /*! \brief Operation failed -    Subject key identifier is invalid**/
#define TLK_CERT_RC_ERROR_X509_UTIL_SUBJECT_KEY_ID_INVALID                 (TLK_CERT_X509_UTIL_MODULE_ERROR_BASE+0x02)

    /*! \brief Operation failed -    Key usage format incorrect or bit 5 (Sign bit) is off**/
#define TLK_CERT_RC_ERROR_X509_UTIL_KEY_USAGE_INVALID                      (TLK_CERT_X509_UTIL_MODULE_ERROR_BASE+0x03)

    /*! \brief Operation failed -    User supplied too small workspace size **/
#define TLK_CERT_RC_ERROR_X509_UTIL_WORKSPACE_SIZE_TOO_SMALL               (TLK_CERT_X509_UTIL_MODULE_ERROR_BASE+0x04)

    /*! \brief Operation failed -    Expected that basicConstraints CA flag is set**/
#define TLK_CERT_RC_ERROR_X509_UTIL_IS_CA_FLAG_INVALID                     (TLK_CERT_X509_UTIL_MODULE_ERROR_BASE+0x05)

    /*! \brief Operation failed -    Subject name in end-entity certificate missing**/
#define TLK_CERT_RC_ERROR_X509_UTIL_EE_SUBJECT_NAME_MISSING                (TLK_CERT_X509_UTIL_MODULE_ERROR_BASE+0x06)

    /*! \brief Operation failed -    Subject name in trusted root certificate missing**/
#define TLK_CERT_RC_ERROR_X509_UTIL_TR_SUBJECT_NAME_MISSING                (TLK_CERT_X509_UTIL_MODULE_ERROR_BASE+0x07)

    /*! \brief Operation failed -    Subject name in certificate missing**/
#define TLK_CERT_RC_ERROR_X509_UTIL_SUBJECT_NAME_MISSING                   (TLK_CERT_X509_UTIL_MODULE_ERROR_BASE+0x08)

    /*! \brief Operation failed -    Issuer name in certificate missing**/
#define TLK_CERT_RC_ERROR_X509_UTIL_USSUER_NAME_MISSING                    (TLK_CERT_X509_UTIL_MODULE_ERROR_BASE+0x09)

    /*! \brief Operation failed -    Key Identifier verification failed**/
#define TLK_CERT_RC_ERROR_X509_UTIL_KEY_IDENTIFIER_VERIFY_FAILD            (TLK_CERT_X509_UTIL_MODULE_ERROR_BASE+0x0A)

    /*! \brief Operation failed -    Problem to convert validNotBefore field from date format to seconds_sinse format **/
#define TLK_CERT_RC_ERROR_X509_UTIL_NOT_BEFORE_TIME_FIELD_FAILED           (TLK_CERT_X509_UTIL_MODULE_ERROR_BASE+0x0B)

    /*! \brief Operation failed -    Problem to convert validNotAfter field from date format to seconds_sinse format **/
#define TLK_CERT_RC_ERROR_X509_UTIL_NOT_AFTER_TIME_FIELD_FAILED            (TLK_CERT_X509_UTIL_MODULE_ERROR_BASE+0x0C)

    /*! \brief Operation failed -    The current certificate is expired **/
#define TLK_CERT_RC_ERROR_X509_UTIL_VALIDITY_PERIOD_VERIFY_FAILED          (TLK_CERT_X509_UTIL_MODULE_ERROR_BASE+0x0D)
    /************************************************************************/
    /* OCSP UTILS Module Errors  - 0xBD000500                               */
    /************************************************************************/

    /*! \brief Operation failed -    OCSP utility failed**/
#define TLK_CERT_RC_ERROR_OCSP_UTIL_FAIL                                   (TLK_CERT_OCSP_UTIL_MODULE_ERROR_BASE+0x01)

    /*! \brief Operation failed -    The response issuer name is invalid **/
#define TLK_CERT_RC_ERROR_OCSP_UTIL_ISSUER_NAME_INVALID                    (TLK_CERT_OCSP_UTIL_MODULE_ERROR_BASE+0x02)

    /*! \brief Operation failed -    Key usage extension invalid (missing or invalid format)**/
#define TLK_CERT_RC_ERROR_OCSP_UTIL_KEY_USAGE_EXTENSION_INVALID            (TLK_CERT_OCSP_UTIL_MODULE_ERROR_BASE+0x03)

    /*! \brief Operation failed -    The number of supplied certificates is over limited **/
#define TLK_CERT_RC_ERROR_OCSP_UTIL_INVALID_NUMBER_OF_CERTIFICATES         (TLK_CERT_OCSP_UTIL_MODULE_ERROR_BASE+0x04)

    /*! \brief Operation failed -    The response does not contains certificate relevant to supplied**/
#define TLK_CERT_RC_ERROR_OCSP_UTIL_NO_PROPER_RESPONDER_CERTIFICATE        (TLK_CERT_OCSP_UTIL_MODULE_ERROR_BASE+0x05)

    /*! \brief Operation failed -    Certificate validity limit verification failed**/
#define TLK_CERT_RC_ERROR_OCSP_UTIL_TIME_VALIDATION_FAILED                 (TLK_CERT_OCSP_UTIL_MODULE_ERROR_BASE+0x06)

    /*! \brief Operation failed -    Response Nonce validation failed**/
#define TLK_CERT_RC_ERROR_OCSP_UTIL_NONCE_VALIDATION_FAILED                (TLK_CERT_OCSP_UTIL_MODULE_ERROR_BASE+0x07)

    /*! \brief Operation failed -    There is no any supplied certificate **/
#define TLK_CERT_RC_ERROR_OCSP_UTIL_NO_SUPPLIED_CERTIFICATES               (TLK_CERT_OCSP_UTIL_MODULE_ERROR_BASE+0x08)

    /*! \brief Operation failed -    Issuer key (HASH) invalid**/
#define TLK_CERT_RC_ERROR_OCSP_UTIL_ISSUER_KEY_INVALID                     (TLK_CERT_OCSP_UTIL_MODULE_ERROR_BASE+0x09)

    /*! \brief Operation failed -    Serial number verification failed**/
#define TLK_CERT_RC_ERROR_OCSP_UTIL_SERIAL_NUM_INVALID                     (TLK_CERT_OCSP_UTIL_MODULE_ERROR_BASE+0x0A)


    /************************************************************************/
    /* Buffer Reader Module Errors - 0xBD000600                             */
    /************************************************************************/
    /*! \brief Operation failed -    Buffer seek operation failed **/
#define TLK_CERT_RC_ERROR_BUFF_SEEK_FAILED                                (TLK_CERT_BUFF_UTIL_MODULE_ERROR_BASE+0x01)

    /*! \brief Operation failed -    Buffer is not big enough**/
#define TLK_CERT_RC_ERROR_BUFF_BUFFER_IS_NOT_BIG_ENOUGH                   (TLK_CERT_BUFF_UTIL_MODULE_ERROR_BASE+0x02)

    /*! \brief Operation failed -    Searching in buffer failed**/
#define TLK_CERT_RC_ERROR_BUFF_ITEM_NOT_FOUND                             (TLK_CERT_BUFF_UTIL_MODULE_ERROR_BASE+0x03)

    /*! \brief Operation failed -    Null pointer received as input**/
#define TLK_CERT_RC_ERROR_BUFF_NULL_POINTER                               (TLK_CERT_BUFF_UTIL_MODULE_ERROR_BASE+0x04)

    /*! \brief Operation failed -    The buffer capacity is over **/
#define TLK_CERT_RC_ERROR_BUFF_NO_MORE_PLACE                              (TLK_CERT_BUFF_UTIL_MODULE_ERROR_BASE+0x05)

    /************************************************************************/
    /* IEEE 1667 Adaptation Module Errors - 0xBD000700                      */
    /************************************************************************/
    /*! \brief Operation failed -    Certificate version is differ from v3 **/
#define TLK_CERT_RC_ERROR_IEEE1667_VERSION_INVALID                        (TLK_CERT_BUFF_1667_MODULE_ERROR_BASE+0x01)
    
    /*! \brief Operation failed -    Action value is incorrect **/
#define TLK_CERT_RC_ERROR_IEEE1667_ACTION_INVALID                         (TLK_CERT_BUFF_1667_MODULE_ERROR_BASE+0x02)

/*! \brief Operation failed -    Missing trusted root certificate subject name **/
#define TLK_CERT_RC_ERROR_IEEE1667_TR_SUBJECT_NAME_MISSING                (TLK_CERT_BUFF_1667_MODULE_ERROR_BASE+0x03)

    /*! \brief Operation failed - Issuer name is missing in certificate **/
#define TLK_CERT_RC_ERROR_IEEE1667_ISSUER_NAME_MISSING                    (TLK_CERT_BUFF_1667_MODULE_ERROR_BASE+0x04)

    /*! \brief Operation failed - Subject name is missing in certificate **/
#define TLK_CERT_RC_ERROR_IEEE1667_SUBJECT_NAME_MISSING                   (TLK_CERT_BUFF_1667_MODULE_ERROR_BASE+0x05)

    /*! \brief Operation failed - Subject name is invalid in certificate **/
#define TLK_CERT_RC_ERROR_IEEE1667_SUBJECT_NAME_INVALID                   (TLK_CERT_BUFF_1667_MODULE_ERROR_BASE+0x06)

    /************************************************************************/
    /* DER Module Errors - 0xBD000800                                       */
    /************************************************************************/
    /*! \brief Operation Warning - not mandatory field is missing **/
#define TLK_CERT_RC_ERROR_DER_WARNING_DER_FIELD_MISSING                   (TLK_CERT_BUFF_DER_MODULE_ERROR_BASE+0x01)
    
    /*! \brief Operation Failed - user allocated not enough memory for output buffer **/ 
#define TLK_CERT_RC_ERROR_DER_NOT_ENOUGH_ROOM_FOR_OUTPUT                  (TLK_CERT_BUFF_DER_MODULE_ERROR_BASE+0x02)

    /*! \brief Operation Failed - one of subject or issuer name fields over its limit **/
#define TLK_CERT_RC_ERROR_DER_FIELD_LENGTH_OVERFLOW                       (TLK_CERT_BUFF_DER_MODULE_ERROR_BASE+0x03)

    /*! \brief Operation Warning - Currently supported maximum field length is overflow **/
    /*   I we got this warning - we should perform appropriate changes  */
#define TLK_CERT_RC_ERROR_DER_WARNING_LENGTH_OVERFLOW                     (TLK_CERT_BUFF_DER_MODULE_ERROR_BASE+0x04)

    /*! \brief Operation Failed - if exits length of length byte in invalid format     **/
    /*  the 8th bit in first byte is set, but now supported 0x81 and 0x82               */
#define TLK_CERT_RC_ERROR_DER_FIELD_LENGTH_FORMAT_INVALID                 (TLK_CERT_BUFF_DER_MODULE_ERROR_BASE+0x05)

    /*! \brief Operation Failed - CRYS Error in calculation signature                   **/
#define TLK_CERT_RC_ERROR_DER_SIGNATURE_CALCULATION                       (TLK_CERT_BUFF_DER_MODULE_ERROR_BASE+0x06)

    /*! \brief Operation Failed - Algorithm signature exists but unsupported            **/
#define TLK_CERT_RC_ERROR_DER_SIGNATURE_AGORITHM_UNSUPPORTED              (TLK_CERT_BUFF_DER_MODULE_ERROR_BASE+0x07)

    /*! \brief Operation Failed - Algorithm id invalid                                  **/
#define TLK_CERT_RC_ERROR_DER_SIGNATURE_AGORITHM_INVALID                  (TLK_CERT_BUFF_DER_MODULE_ERROR_BASE+0x08)

#define TLK_CERT_RC_ERROR_DER_BUILD_PRIVATE_KEY                           (TLK_CERT_BUFF_DER_MODULE_ERROR_BASE+0x08)

#ifdef __cplusplus
}
#endif 

#endif
